<?php

namespace App\Http\Controllers\Dashboard;

use App\Http\Controllers\Controller;
use App\Models\Bank;
use App\Models\PassportHolderCategory;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;

class BankControllerV2 extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        $query = Bank::select('banks.*')->orderBy('banks.name', 'asc');

        if ($request->filled('name')) {
            $query->where('banks.name', $request->name);
        }

        if ($request->filled('status')) {
            $query->whereDate('banks.status', $request->status);
        }

        $datas = $query->paginate(20);

        return view('banks.index', compact(
            'datas'
        ));
    }


    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        return view('banks.create-modal');
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'branch_name' => 'nullable|string|max:255',
            'account_name' => 'required|string|max:255',
            'account_type' => 'required|in:savings,current,fixed',
            'bank_type' => 'required|in:national,international',
            'type' => 'required|in:bank,mobile_banking,digital_wallet',
            'routing_number' => 'nullable|string|max:20',
            'account_number' => 'required|string|max:20|unique:banks,account_number',
            'iban' => 'nullable|string|max:34',
            'swift_code' => 'nullable|string|max:11',
            'currency' => 'required|string|max:10',
            'address' => 'nullable|string|max:255',
            'balance' => 'required|numeric|min:0',
        ]);

        // If validation fails
        if ($validator->fails()) {
            if ($request->ajax()) {
                return response()->json([
                    'success' => false,
                    'message' => $validator->errors()->first()
                ]);
            }

            return redirect()->back()->withErrors($validator)->withInput();
        }

        $data = Bank::create([
            'name' => $request->name,
            'branch_name' => $request->branch_name,
            'account_name' => $request->account_name,
            'account_type' => $request->account_type,
            'bank_type' => $request->bank_type,
            'type' => $request->type,
            'routing_number' => $request->routing_number,
            'account_number' => $request->account_number,
            'iban' => $request->iban,
            'swift_code' => $request->swift_code,
            'currency' => $request->currency,
            'address' => $request->address,
            'balance' => $request->balance,
            'last_transaction_date' => $request->last_transaction_date,
            'last_transaction_amount' => $request->last_transaction_amount,
            'last_transaction_type' => $request->last_transaction_type,
            'created_by' => auth()->id(),
            'status' => $request->status ? 1 : 0
        ]);

        if ($request->ajax()) {
            return response()->json([
                'success' => true,
                'message' => 'Data created successfully.',
                'data' => $data
            ]);
        }

        return redirect()->route('role.banks.index')->with('success', 'Data created successfully.');
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        return view('banks.edit-modal', compact('id'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        $id = $request->id;
        $data = Bank::findOrFail($id);
        if (empty($data)) {
            return response()->json([
                'success' => false,
                'message' => 'Data Info Not Found!'
            ]);
        }
        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'branch_name' => 'nullable|string|max:255',
            'account_name' => 'required|string|max:255',
            'account_type' => 'required|in:savings,current,fixed',
            'bank_type' => 'required|in:national,international',
            'type' => 'required|in:bank,mobile_banking,digital_wallet',
            'routing_number' => 'nullable|string|max:20',
            'account_number' => 'required|string|max:20|unique:banks,account_number,' . $data->id,
            'iban' => 'nullable|string|max:34',
            'swift_code' => 'nullable|string|max:11',
            'currency' => 'required|string|max:10',
            'address' => 'nullable|string|max:255',
            'balance' => 'required|numeric|min:0',
        ]);

        // If validation fails
        if ($validator->fails()) {
            if ($request->ajax()) {
                return response()->json([
                    'success' => false,
                    'message' => $validator->errors()->first()
                ]);
            }
            return redirect()->back()->withErrors($validator)->withInput();
        }

        $data->update([
            'name' => $request->name,
            'branch_name' => $request->branch_name,
            'account_name' => $request->account_name,
            'account_type' => $request->account_type,
            'bank_type' => $request->bank_type,
            'type' => $request->type,
            'routing_number' => $request->routing_number,
            'account_number' => $request->account_number,
            'iban' => $request->iban,
            'swift_code' => $request->swift_code,
            'currency' => $request->currency,
            'address' => $request->address,
            'balance' => $request->balance,
            'last_transaction_date' => $request->last_transaction_date,
            'last_transaction_amount' => $request->last_transaction_amount,
            'last_transaction_type' => $request->last_transaction_type,
            'updated_by' => auth()->id(),
            'status' => $request->status ? 1 : 0
        ]);

        if ($request->ajax()) {
            return response()->json([
                'success' => true,
                'message' => 'Data updated successfully.',
                'data' => $data
            ]);
        }

        return redirect()->back()->with('success', 'Data updated successfully.');
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy(Request $request, $id)
    {
        try {
            $item = Bank::find($request->item_id);
            if ($item) {
                $item->delete();
            } else {
                return response()->json([
                    'success' => false,
                    'message' => 'Data Info Not Found!'
                ]);
            }
        } catch (\Throwable $th) {
            return response()->json([
                'success' => false,
                'message' => $th->getMessage()
            ]);
        }

        return response()->json([
            'success' => true,
            'message' => 'Data deleted successfully.'
        ]);
    }

    public function getEmployeeSalary(Request $request)
    {
        try {
            $data = Bank::where('user_id', $request->user_id)->orderBy('id', 'ASC')->get();
        } catch (\Throwable $th) {
            return response()->json([
                'success' => false,
                'message' => $th->getMessage()
            ]);
        }

        return response()->json([
            'data' => $data,
            'success' => true,
            'message' => 'Data Found Successfully.'
        ]);
    }
}
